package com.afanticar.afantiopenapi.service;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.date.DateUtil;
import cn.hutool.core.util.StrUtil;
import com.afanticar.afantiopenapi.mapper.DwsAfantiAdbDataOssRecordMapper;
import com.afanticar.afantiopenapi.mapper.OpenApiClientPrincipalRelationMapper;
import com.afanticar.afantiopenapi.model.dto.CommonPageInfoDTO;
import com.afanticar.afantiopenapi.model.dto.DataReportRequestDTO;
import com.afanticar.afantiopenapi.model.dto.PageInfoDTO;
import com.afanticar.afantiopenapi.model.dto.ReportDataInfoDTO;
import com.afanticar.afantiopenapi.model.entity.DwsAfantiAdbDataOssRecordEntity;
import com.afanticar.afantiopenapi.model.entity.OpenApiClientPrincipalRelationEntity;
import com.afanticar.common.core.api.ResultCodeEnum;
import com.afanticar.common.core.exception.ApiException;
import com.aliyun.oss.OSS;
import com.aliyun.oss.OSSClientBuilder;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.collect.Lists;
import lombok.SneakyThrows;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;

import javax.annotation.PostConstruct;
import javax.annotation.Resource;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;
import java.time.Instant;
import java.time.LocalDate;
import java.util.Date;
import java.util.List;
import java.util.Objects;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * @author hejincai
 * @since 2025/4/24 9:41
 **/
@Service
@Slf4j
public class DataReportService {

    @Resource
    private OpenApiClientPrincipalRelationMapper openApiClientPrincipalRelationMapper;

    @Resource
    private DwsAfantiAdbDataOssRecordMapper dwsAfantiAdbDataOssRecordMapper;

    @Resource
    private ObjectMapper objectMapper;

    @Value("${spring.rocket-mq.accessKey}")
    private String ossAccessKey;

    @Value("${spring.rocket-mq.secretKey}")
    private String ossAccessSecret;

    private OSS oss;

    @PostConstruct
    public void init() {
        String ossEndpoint = "oss-cn-hangzhou.aliyuncs.com";
        oss = new OSSClientBuilder().build(ossEndpoint, ossAccessKey, ossAccessSecret);
    }

    @SneakyThrows
    public CommonPageInfoDTO<ReportDataInfoDTO> report(String clientId, DataReportRequestDTO requestBody) {
        log.info("请求数据报表入参，client_id:{}, 请求参数：{}", clientId, objectMapper.writeValueAsString(requestBody));
        handleTimeRange(requestBody);
        List<OpenApiClientPrincipalRelationEntity> openApiClientPrincipalEntityList = openApiClientPrincipalRelationMapper.selectByClientId(clientId);
        if (CollUtil.isEmpty(openApiClientPrincipalEntityList)) {
            throw new ApiException("客户端未配置品牌，请联系管理员");
        }
        OpenApiClientPrincipalRelationEntity openApiClientPrincipalRelationEntity = openApiClientPrincipalEntityList.get(0);
        String principalId = openApiClientPrincipalRelationEntity.getPrincipalId();
        requestBody.setPrincipalId(principalId);
        Page<DwsAfantiAdbDataOssRecordEntity> page = dwsAfantiAdbDataOssRecordMapper.page(new Page<>(requestBody.getPage(), requestBody.getSize()), requestBody);
        return convertToCommonPageInfo(page);
    }

    private void handleTimeRange(DataReportRequestDTO requestBody) {
        LocalDate yesterday = LocalDate.now().minusDays(1);
        if (Objects.nonNull(requestBody.getStartTime())) {
            LocalDate sevenDaysAgo = LocalDate.now().minusDays(7);
            if (sevenDaysAgo.isAfter(requestBody.getStartTime())) {
                throw new ApiException(ResultCodeEnum.VALIDATE_FAILED.getCode(), "开始时间不能早于前7天");
            }
        }
        if (Objects.isNull(requestBody.getStartTime()) && Objects.isNull(requestBody.getEndTime())) {
            requestBody.setStartTime(yesterday);
            requestBody.setEndTime(yesterday);
        } else if (Objects.isNull(requestBody.getStartTime()) && Objects.nonNull(requestBody.getEndTime())) {
            requestBody.setStartTime(requestBody.getEndTime());
        } else if (Objects.isNull(requestBody.getEndTime()) && Objects.nonNull(requestBody.getStartTime())) {
            requestBody.setEndTime(requestBody.getStartTime());
        } else {

            if (requestBody.getEndTime().isBefore(requestBody.getStartTime())) {
                throw new ApiException(ResultCodeEnum.VALIDATE_FAILED.getCode(), "开始时间不能晚于结束时间");
            }
        }
    }


    @SneakyThrows
    private String generatePresignedUrl(String ossUrl) {
        // 生成一个带过期时间的url
        String[] ossObjInfo = parseOSSUrl(ossUrl);
        Instant expireInstant = Instant.now().plusSeconds(86400);
        URL url = oss.generatePresignedUrl(ossObjInfo[0], ossObjInfo[1], Date.from(expireInstant));
        return url.toString();
    }

    public static String[] parseOSSUrl(String ossUrl) throws URISyntaxException {
        // 处理URL编码的特殊字符（如%2F）
        // 使用正则表达式匹配bucket和object
        Pattern pattern = Pattern.compile("https?://([^.]+)\\.oss(-[^.]+)?\\.([^/]+)/(.+)");
        Matcher matcher = pattern.matcher(ossUrl);

        if (matcher.find()) {
            String bucket = matcher.group(1);
            // 从匹配结果中提取object，并去掉查询参数
            String fullObjectPath = matcher.group(4);
            String object = fullObjectPath.split("\\?")[0];
            return new String[]{bucket, object};
        }

        // 如果正则匹配失败，尝试URI解析
        URI uri = new URI(ossUrl);
        String host = uri.getHost();

        if (host != null && host.endsWith(".aliyuncs.com")) {
            String bucket = host.split("\\.")[0];
            String object = uri.getPath().substring(1); // 去掉开头的/
            return new String[]{bucket, object};
        }

        throw new IllegalArgumentException("无法从URL中解析出OSS bucket和object: " + ossUrl);
    }


    private CommonPageInfoDTO<ReportDataInfoDTO> convertToCommonPageInfo(Page<DwsAfantiAdbDataOssRecordEntity> page) {

        CommonPageInfoDTO<ReportDataInfoDTO> commonPageInfo = new CommonPageInfoDTO<>();
        PageInfoDTO pageInfoDTO = new PageInfoDTO();
        pageInfoDTO.setPage((int) page.getCurrent());
        pageInfoDTO.setSize((int) page.getSize());
        pageInfoDTO.setTotal((int) page.getTotal());
        pageInfoDTO.setTotalPage((int) page.getPages());
        commonPageInfo.setPageInfo(pageInfoDTO);
        List<DwsAfantiAdbDataOssRecordEntity> entityList = page.getRecords();
        List<ReportDataInfoDTO> list = Lists.newArrayList();
        for (DwsAfantiAdbDataOssRecordEntity entity : entityList) {
            ReportDataInfoDTO reportDataInfo = new ReportDataInfoDTO();
            reportDataInfo.setStatisticsDay(entity.getStatisticsDay().toLocalDate());
            reportDataInfo.setPlatform(entity.getPlatform());
            reportDataInfo.setDataType(entity.getDataType());
            reportDataInfo.setFileUrl(generatePresignedUrl(entity.getOssUrl()));
            reportDataInfo.setCtime(entity.getCtime());
            reportDataInfo.setMtime(entity.getMtime());
            reportDataInfo.setId(StrUtil.format("{}:{}:{}:{}", entity.getPrincipalId(),
                    DateUtil.format(entity.getStatisticsDay(), "yyyy-MM-dd"),
                    entity.getPlatform(), entity.getDataType()));
            list.add(reportDataInfo);
        }
        commonPageInfo.setRows(list);
        return commonPageInfo;

    }
}
